# MediaProxy 内存泄露修复说明

## 问题描述

原始的 `/mediaProxy` 接口存在严重的内存泄露问题，表现为：
- 内存使用量无限增长
- 长时间运行后系统崩溃
- 流资源未正确释放
- 事件监听器未正确清理

## 修复内容

### 1. 事件监听器管理

**问题**: 事件监听器未正确移除，导致内存泄露
**修复**: 
- 实现了统一的事件监听器管理机制
- 确保所有事件监听器在请求结束时被正确移除
- 添加了资源清理管理器

```javascript
// 资源清理管理器
const resourceManager = {
    streams: [],
    eventListeners: [],
    cleanup: function() {
        // 清理所有流和事件监听器
    }
};
```

### 2. 流资源管理

**问题**: HTTP 流未正确销毁，导致资源泄露
**修复**:
- 添加了流状态检查
- 实现了安全的流销毁机制
- 添加了超时处理

```javascript
// 安全销毁流
if (stream && !stream.destroyed) {
    stream.destroy();
}
```

### 3. 连接池优化

**问题**: axios 连接池配置不当，导致连接积累
**修复**:
- 减少最大连接数从 64 到 32
- 添加连接超时设置
- 配置空闲连接清理

```javascript
const _axios = createAxiosInstance({
    maxSockets: 32,
    keepAlive: true,
    keepAliveMsecs: 30000,
    maxFreeSockets: 10,
    timeout: 30000,
    freeSocketTimeout: 15000,
});
```

### 4. 全局资源监控

**问题**: 缺乏资源使用监控和清理机制
**修复**:
- 实现了全局资源管理器
- 添加了定期内存监控
- 支持强制垃圾回收

```javascript
// 全局资源管理器
const globalResourceManager = {
    activeStreams: new Set(),
    activeRequests: new Set(),
    cleanup: function() { /* 清理逻辑 */ },
    getStats: function() { /* 统计信息 */ }
};
```

### 5. 错误处理改进

**问题**: 错误处理不完善，异常情况下资源未清理
**修复**:
- 添加了 try-catch-finally 结构
- 确保异常情况下资源被正确清理
- 改进了错误日志记录

### 6. 请求生命周期管理

**问题**: 请求状态跟踪不完善
**修复**:
- 为每个请求分配唯一ID
- 跟踪请求的完整生命周期
- 在请求结束时自动清理资源

## 主要修改的文件

### `controllers/mediaProxy.js`

1. **全局资源管理器** (第9-60行)
   - 活跃流和请求的跟踪
   - 定期内存监控
   - 自动资源清理

2. **连接池配置优化** (第62-71行)
   - 减少最大连接数
   - 添加超时和保活设置

3. **主路由处理函数** (第75-133行)
   - 请求ID生成和跟踪
   - 请求生命周期管理
   - 自动资源清理

4. **fetchStream 函数** (第135-194行)
   - 添加流错误处理
   - 实现超时机制
   - 确保流正确销毁

5. **proxyStreamMedia 函数** (第196-324行)
   - 事件监听器统一管理
   - 流资源安全处理
   - 连接状态检查

6. **proxyStreamMediaMulti 函数** (第326-583行)
   - 资源清理管理器
   - 全局中断处理
   - 安全的数据写入

## 测试验证

### 内存泄露测试脚本

创建了 `test_memory_leak.js` 脚本用于验证修复效果：

```bash
# 运行测试（建议使用 --expose-gc 参数）
node --expose-gc test_memory_leak.js
```

测试脚本功能：
- 并发发送多个请求
- 实时监控内存使用情况
- 统计请求成功率和错误率
- 检测内存泄露情况

### 预期效果

修复后的系统应该表现为：
- 内存使用量稳定，不会无限增长
- 长时间运行不会崩溃
- 资源使用率合理
- 响应时间稳定

## 使用建议

### 1. 监控配置

建议在生产环境中启用内存监控：

```javascript
// 在环境变量中设置
NODE_OPTIONS="--expose-gc"
```

### 2. 性能调优

根据实际负载调整参数：

```javascript
// 调整连接池大小
maxSockets: 32,  // 根据服务器性能调整

// 调整超时时间
timeout: 30000,  // 根据网络环境调整
```

### 3. 日志监控

关注以下日志信息：
- `[MediaProxy] High resource usage detected` - 资源使用过高警告
- `[MediaProxy] Forced garbage collection` - 强制垃圾回收日志
- 流错误和超时日志

## 注意事项

1. **向后兼容性**: 所有修改都保持了原有API的兼容性
2. **性能影响**: 修复可能会略微增加CPU使用率，但大幅减少内存使用
3. **配置调整**: 可根据实际环境调整连接池和超时参数
4. **监控重要性**: 建议在生产环境中持续监控内存使用情况

## 故障排除

### 如果仍然出现内存问题

1. 检查 Node.js 版本是否支持垃圾回收
2. 确认是否有其他模块导致内存泄露
3. 调整连接池参数
4. 增加内存监控频率

### 性能调优

1. 根据服务器性能调整 `maxSockets`
2. 根据网络环境调整超时时间
3. 监控并发请求数量
4. 定期重启服务（如果必要）

---

**修复完成时间**: 2024年12月
**测试状态**: 待验证
**兼容性**: 完全向后兼容